#include <std.glsl>
#ifndef KERNEL_SIZE
#define KERNEL_SIZE 16
#endif
#if (SC_DEVICE_CLASS<SC_DEVICE_CLASS_C) && (KERNEL_SIZE>9)
#define KERNEL_SIZE_TUNED 9
#else
#define KERNEL_SIZE_TUNED KERNEL_SIZE
#endif
#ifndef IMAGE_HEIGHT
#define IMAGE_HEIGHT 1280.0
#endif
#ifndef IMAGE_WIDTH
#define IMAGE_WIDTH 720.0
#endif
#ifndef MIX_AMOUNT
#define MIX_AMOUNT 0.01
#endif

varying vec2 texCoordinates[KERNEL_SIZE_TUNED];
uniform sampler2D mainTexture;
uniform sampler2D accumulatorTexture;
uniform mat3 mainTextureTransform;
uniform float firstRun;
uniform float mixSpeed;

uniform vec3 face1;
uniform vec2 face1dir;

#ifdef SECOND_FACE
uniform vec3 face2;
uniform vec2 face2dir;
#endif

uniform sampler2D texMask;
uniform float flipMaskVertically;

#ifdef FACE_ONLY
#define faceOffset-0.02
#define faceRadius 0.4
#define faceAspect 1.4
#else
#define faceOffset 0.0
#define faceRadius 0.4725
#define faceAspect 1.4
#endif

#ifdef VERTEX_SHADER
#if KERNEL_SIZE_TUNED==9
void setCoordinates(vec2 texCoords){
	texCoordinates[0]=texCoords+vec2(-0.166667,-0.166667);
	texCoordinates[1]=texCoords+vec2(0.000000,-0.166667);
	texCoordinates[2]=texCoords+vec2(0.166667,-0.166667);
	texCoordinates[3]=texCoords+vec2(-0.166667,0.000000);
	texCoordinates[4]=texCoords+vec2(0.000000,0.000000);
	texCoordinates[5]=texCoords+vec2(0.166667,0.000000);
	texCoordinates[6]=texCoords+vec2(-0.166667,0.166667);
	texCoordinates[7]=texCoords+vec2(0.000000,0.166667);
	texCoordinates[8]=texCoords+vec2(0.166667,0.166667);
}
#endif
#if KERNEL_SIZE_TUNED==16
void setCoordinates(vec2 texCoords){
	texCoordinates[0]=texCoords+vec2(-0.187500,-0.187500);
	texCoordinates[1]=texCoords+vec2(-0.062500,-0.187500);
	texCoordinates[2]=texCoords+vec2(0.062500,-0.187500);
	texCoordinates[3]=texCoords+vec2(0.187500,-0.187500);
	texCoordinates[4]=texCoords+vec2(-0.187500,-0.062500);
	texCoordinates[5]=texCoords+vec2(-0.062500,-0.062500);
	texCoordinates[6]=texCoords+vec2(0.062500,-0.062500);
	texCoordinates[7]=texCoords+vec2(0.187500,-0.062500);
	texCoordinates[8]=texCoords+vec2(-0.187500,0.062500);
	texCoordinates[9]=texCoords+vec2(-0.062500,0.062500);
	texCoordinates[10]=texCoords+vec2(0.062500,0.062500);
	texCoordinates[11]=texCoords+vec2(0.187500,0.062500);
	texCoordinates[12]=texCoords+vec2(-0.187500,0.187500);
	texCoordinates[13]=texCoords+vec2(-0.062500,0.187500);
	texCoordinates[14]=texCoords+vec2(0.062500,0.187500);
	texCoordinates[15]=texCoords+vec2(0.187500,0.187500);
}
#endif

#define SCALE_TO_FACE
#ifdef SCALE_TO_FACE
void setCoordinatesToFace1(vec2 texCoords){
	const vec2 imageDim=vec2(IMAGE_WIDTH,IMAGE_HEIGHT);
	if (face1.z>0.0){
		for (int n=0;n<KERNEL_SIZE_TUNED;++n){
			vec2 texC=face1.z*(texCoordinates[n]-0.5)+vec2(face1.x,1.0-face1.y);
			texCoordinates[n]=floor(texC*imageDim+0.5)/imageDim;
		}
	}
}
#endif

attribute vec4 position;
void main(){
	vec4 positionScr=sc_ModelViewProjectionMatrix*position;
	setCoordinates(0.5*positionScr.xy+0.5);
#ifdef SCALE_TO_FACE
	setCoordinatesToFace1(0.5*positionScr.xy+0.5);
#endif
	gl_Position=positionScr;
}
#endif
#ifdef FRAGMENT_SHADER
float facestep(vec3 face,vec2 dir,vec2 pos)
{
	float size=face.z;
	vec2 faceCenter=face.xy;
	faceCenter.y+=faceOffset;

	vec2 faceToCoords=(pos-faceCenter);
	faceToCoords.x*=sc_Camera.aspect;
	faceToCoords=vec2(dot(faceToCoords,dir)*faceAspect,dot(faceToCoords,vec2(-dir.y,dir.x)));

	return step(faceRadius*size,length(faceToCoords));
}

float relatePos(vec2 pos)
{
	float alpha=1.0;

#if defined(FACE_EXCLUDE) || defined(FACE_ONLY)
	if (face1.z>0.0)
	{
		alpha=facestep(face1,face1dir,pos);
#ifdef SECOND_FACE
		if (face2.z>0.0)
		{
			alpha=min(alpha,facestep(face2,face2dir,pos));
		}
#endif

#ifdef FACE_ONLY
		alpha=1.0-alpha;
#endif
	}
#endif

#ifdef ENABLE_MASK_TEX
	if (flipMaskVertically>0.0){
		pos.y=1.0-pos.y;
	}
	alpha=min(alpha,texture2D(texMask,pos).r);
#endif
	return alpha;
}

#ifdef MEDIAN
#define fragCount KERNEL_SIZE_TUNED
vec4 frags[fragCount];
void sort1(int i)
{
	vec4 t;
	for (int j=i+1;j<fragCount;++j){
		if (frags[j].a<frags[i].a){
			t=frags[i];
			frags[i]=frags[j];
			frags[j]=t;
		}
	}
}
void sortFragments()
{
	sort1(0);
	sort1(1);
	sort1(2);
	sort1(3);
	sort1(4);
#if KERNEL_SIZE_TUNED==16
	sort1(5);
	sort1(6);
	sort1(7);
	sort1(8);
#endif
}
#if defined(FACE_EXCLUDE) || defined(FACE_ONLY) || defined(ENABLE_MASK_TEX)
vec4 average(sampler2D tex){
	vec3 texColor;
	float kernel_sum=0.0;
	for (int n=0;n<KERNEL_SIZE_TUNED;++n){
		float alpha=relatePos(texCoordinates[n]);
		kernel_sum+=alpha;
		texColor=mix(vec3(1.0),texture2D(tex,texCoordinates[n]).rgb,alpha);
		frags[n]=vec4(texColor,dot(texColor,vec3(0.299,0.587,0.114)));
	}
	sortFragments();
	int iKernel=int(kernel_sum);
	return vec4(frags[iKernel/2].rgb,sqrt(kernel_sum/float(KERNEL_SIZE_TUNED)));
}
#else
vec4 average(sampler2D tex){
	vec3 texColor;
	for (int n=0;n<KERNEL_SIZE_TUNED;++n){
		texColor=texture2D(tex,texCoordinates[n]).rgb;
		frags[n]=vec4(texColor,dot(texColor,vec3(0.299,0.587,0.114)));
	}
	sortFragments();
	return vec4(frags[KERNEL_SIZE_TUNED/2].rgb,1.0);
}
#endif
#else
#if defined(FACE_EXCLUDE) || defined(FACE_ONLY) || defined(ENABLE_MASK_TEX)
vec4 average(sampler2D tex){
	vec3 avgColor=vec3(0.0);
	float kernel_sum=0.0;
	for (int n=0;n<KERNEL_SIZE_TUNED;++n){
		float alpha=relatePos(texCoordinates[n]);
		kernel_sum+=alpha;
		avgColor+=alpha*texture2D(tex,texCoordinates[n]).rgb;
	}

	return vec4(avgColor/max(kernel_sum,1.0),sqrt(kernel_sum/float(KERNEL_SIZE_TUNED)));
}
#else
vec4 average(sampler2D tex){
	vec3 avgColor=vec3(0.0);
	for (int n=0;n<KERNEL_SIZE_TUNED;++n){
		avgColor+=texture2D(tex,texCoordinates[n]).rgb;
	}
	return vec4(avgColor/float(KERNEL_SIZE_TUNED),1.0);
}
#endif
#endif

void main()
{
	vec4 data=average(mainTexture);
	float t=clamp(mixSpeed*data.a,firstRun,1.0);

	gl_FragColor=vec4(mix(texture2D(accumulatorTexture,vec2(0.5)).rgb,data.rgb,t),1.0);
}
#endif

#if sc_IsEditor
#error This is an exported shader. Please do not use shaders in Studio that have already been exported to a lens! Only use fresh shaders,presets,or shaders from existing Studio projects!
#endif

/// Exported with Lens Studio 3.2.0.0 Internal
